import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Camera, Star, Calendar, Video, MessageSquare } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyBookings.css'

const MyBookings = () => {
  const [bookings, setBookings] = useState([])

  useEffect(() => {
    loadBookings()
    const interval = setInterval(loadBookings, 500)
    return () => clearInterval(interval)
  }, [])

  const loadBookings = () => {
    const saved = localStorage.getItem('studioBookings')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setBookings(parsed)
      } catch (e) {
        setBookings([])
      }
    } else {
      const defaultBookings = [
        {
          id: 1,
          studio: 'StudioPro Premium',
          date: '2025-03-20',
          equipment: 'Профессиональное освещение, камеры',
          review: 'Отличная студия, все оборудование работает идеально!'
        },
        {
          id: 2,
          studio: 'StudioPro Chroma',
          date: '2025-03-15',
          equipment: 'Хромакей, студийное освещение',
          review: 'Прекрасные условия для съемки'
        },
        {
          id: 3,
          studio: 'StudioPro Interior',
          date: '2025-03-10',
          equipment: 'Интерьерная студия, естественный свет',
          review: 'Очень уютная атмосфера'
        }
      ]
      setBookings(defaultBookings)
      localStorage.setItem('studioBookings', JSON.stringify(defaultBookings))
    }
  }

  const totalBookings = bookings.length
  const totalStudios = new Set(bookings.map(b => b.studio)).size
  const averageRating = 4.8

  return (
    <div className="my-bookings">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="hero-icon-wrapper"
            initial={{ scale: 0, rotate: -180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Camera size={100} strokeWidth={2.5} className="hero-icon" />
          </motion.div>
          <h1 className="hero-title">Мои бронирования</h1>
          <p className="hero-description">
            История ваших аренд студий и творческих проектов
          </p>
        </motion.section>

        <motion.div 
          className="stats-grid"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <motion.div 
            className="stat-card"
            whileHover={{ y: -4, transition: { duration: 0.2 } }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(255, 107, 53, 0.2)' }}>
              <Video size={48} strokeWidth={2.5} color="#FF6B35" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{totalBookings}</div>
              <div className="stat-label">Бронирований</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ y: -4, transition: { duration: 0.2 } }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(247, 147, 30, 0.2)' }}>
              <Star size={48} strokeWidth={2.5} color="#F7931E" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{averageRating}</div>
              <div className="stat-label">Средняя оценка</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ y: -4, transition: { duration: 0.2 } }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(255, 140, 66, 0.2)' }}>
              <Camera size={48} strokeWidth={2.5} color="#FF8C42" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{totalStudios}</div>
              <div className="stat-label">Студий</div>
            </div>
          </motion.div>
        </motion.div>

        <ImageSlider images={[
          '/images/studio-1.jpg',
          '/images/studio-2.jpg',
          '/images/studio-3.jpg',
          '/images/studio-4.jpg'
        ]} />

        <motion.section 
          className="bookings-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title">Ваши бронирования</h2>
          {bookings.length > 0 ? (
            <div className="bookings-grid">
              {bookings.map((booking, index) => (
                <motion.div 
                  key={booking.id} 
                  className="booking-card"
                  initial={{ opacity: 0, scale: 0.9, rotate: -5 }}
                  animate={{ opacity: 1, scale: 1, rotate: 0 }}
                  transition={{ delay: index * 0.15, duration: 0.5 }}
                  whileHover={{ 
                    y: -4,
                    transition: { duration: 0.2 }
                  }}
                  style={{ 
                    borderLeft: '6px solid var(--primary)',
                    background: 'linear-gradient(135deg, rgba(26, 26, 26, 0.99) 0%, rgba(20, 20, 20, 0.98) 100%)'
                  }}
                >
                  <div className="booking-header">
                    <div className="booking-studio-badge" style={{ background: 'rgba(255, 107, 53, 0.25)', color: '#FF6B35' }}>
                      {booking.studio}
                    </div>
                    <div className="booking-rating">
                      {Array.from({ length: 5 }, (_, i) => (
                        <Star
                          key={i}
                          size={20}
                          fill={i < 4 ? '#F59E0B' : 'none'}
                          stroke={i < 4 ? '#F59E0B' : '#666666'}
                          strokeWidth={2}
                        />
                      ))}
                    </div>
                  </div>
                  <div className="booking-content">
                    <div className="booking-info-item">
                      <Calendar size={24} strokeWidth={2.5} color="#FF6B35" />
                      <span className="booking-info-text">
                        {new Date(booking.date).toLocaleDateString('ru-RU', { 
                          day: 'numeric', 
                          month: 'long', 
                          year: 'numeric' 
                        })}
                      </span>
                    </div>
                    <div className="booking-info-item">
                      <Video size={24} strokeWidth={2.5} color="#FF6B35" />
                      <span className="booking-info-text">{booking.equipment}</span>
                    </div>
                    <div className="booking-review">
                      <MessageSquare size={20} strokeWidth={2} color="#F7931E" />
                      <span className="booking-review-text">{booking.review}</span>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <div className="empty-icon-circle">
                <Camera size={80} strokeWidth={1.5} />
              </div>
              <h3 className="empty-title">Нет бронирований</h3>
              <p className="empty-text">Забронируйте студию для вашего проекта прямо сейчас</p>
            </div>
          )}
        </motion.section>

      </div>
    </div>
  )
}

export default MyBookings

